using log4net;
using Microsoft.Azure.WebJobs;
using Microsoft.Azure.WebJobs.Extensions.Http;
using System;
using System.Net;
using System.Net.Http;
using System.Threading.Tasks;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.OrchestrationService.Interface;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.Functions
{
    public static class CcnDataStream
    {
        [FunctionName("CcnDataStream")]
        public static async Task<HttpResponseMessage> Run([HttpTrigger(AuthorizationLevel.Anonymous, "post", Route = "CcnDataStream/v1")]HttpRequestMessage req,
            [Inject]IOrchestrationService orchestrationService,
            [Inject] ILog logger)
        {
            try
            {
                logger.Info("@@@@@ Start CcnDataStream http function @@@@@");

                var headerValues = HttpUtility.ParseHeaders(req);
                logger.Info($"---- CcnDataStream - Conversation: {headerValues.ConversationId}; {headerValues.SenderId}");

                var payload = await req.Content.ReadAsStringAsync();
                var providers = Utilities.Deserialize<Providers>(payload);

                if (providers == null)
                {
                    logger.Error("CcnDataStream: Provider payload is missing");
                    return req.CreateErrorResponse(HttpStatusCode.BadRequest, "Provider payload is missing");
                }
                
                providers.TransactionId = headerValues.TransactionId;
                providers.NetworkId = headerValues.SenderId;

                if (!string.IsNullOrEmpty(providers.TransactionId) && !string.IsNullOrEmpty(providers.NetworkId) && (providers.Provider != null) && providers.Provider.Count > 0)
                {
                    await orchestrationService.StartProcessingAsync(headerValues, providers);

                    logger.Info("@@@@@ End CcnDataStream http function @@@@@");

                    return req.CreateResponse(HttpStatusCode.Created);
                }

                logger.Error("CcnDataStream: Provider is missing data required to process the transaction");
                return req.CreateErrorResponse(HttpStatusCode.BadRequest, "Provider is missing data required to process the transaction");
            }
            catch (PpmsSerializationException ex)
            {
                logger.Error("An error occurred deserializing the provider data: ", ex);
                return req.CreateErrorResponse(HttpStatusCode.BadRequest, "An serialization error occurred in the CcnDataStream Function API: ", ex);
            }
            catch (Exception ex)
            {
                logger.Error("An error occurred in the CcnDataStream Function API: ", ex);
                return req.CreateErrorResponse(HttpStatusCode.InternalServerError, "An error occurred in the CcnDataStream Function API: ", ex);
            }
        }
    }
}